//+------------------------------------------------------------------+
//|                                                    Hunter_EA.mq5 |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#property version   "1.00"
#property description "Hunter EA - FVG and Liquidity-based trading system"
#property description "Single-file version for easy deployment"

//+------------------------------------------------------------------+
//| ENUMERATIONS AND STRUCTURES (from libraries)                     |
//+------------------------------------------------------------------+

//--- Liquidity types
enum ENUM_LIQUIDITY_TYPE
{
   LIQUIDITY_SWING_HIGH,
   LIQUIDITY_SWING_LOW,
   LIQUIDITY_EQUAL_HIGHS,
   LIQUIDITY_EQUAL_LOWS,
   LIQUIDITY_SESSION_HIGH,
   LIQUIDITY_SESSION_LOW,
   LIQUIDITY_PDH,
   LIQUIDITY_PDL,
   LIQUIDITY_PWH,
   LIQUIDITY_PWL
};

//--- Session types
enum ENUM_SESSION_TYPE
{
   SESSION_SYDNEY,
   SESSION_TOKYO,
   SESSION_LONDON,
   SESSION_NEW_YORK,
   SESSION_SILVER_BULLET_1,
   SESSION_SILVER_BULLET_2,
   SESSION_SILVER_BULLET_3
};

//--- FVG structure
struct FVGInfo
{
   int fvg_id;
   int middle_candle_bar;
   datetime candle_time;
   double top_price;
   double bottom_price;
   double enc_point;
   double gap_size;
   bool is_bullish;
   bool is_active;
   bool enc_touched;
   string directional_bias;
   ulong trade_ticket;
   bool trade_opened;
   datetime enc_time;
   bool is_inverse;
   bool is_reset;
};

//--- Liquidity structure
struct LiquidityInfo
{
   int liquidity_id;
   ENUM_LIQUIDITY_TYPE type;
   datetime time_created;
   double price_level;
   bool is_swept;
   bool is_active;
   datetime sweep_time;
   string session_origin;
   bool is_equal_level;
   int equal_count;
   double tolerance;
};

//--- Session structure
struct SessionData
{
   ENUM_SESSION_TYPE session_type;
   datetime session_start;
   datetime session_end;
   double session_high;
   double session_low;
   bool is_active;
   string session_name;
};

//--- Daily bias structure
struct DailyBiasInfo
{
   datetime bias_date;
   double pdh_level;
   double pdl_level;
   double pwh_level;
   double pwl_level;
   string current_dol;
   string daily_bias;
   bool pdh_failure_to_displace;
   bool pdl_failure_to_displace;
};

//+------------------------------------------------------------------+
//| FVG LIBRARY CLASS                                                |
//+------------------------------------------------------------------+
class CFVGLibrary
{
private:
   FVGInfo           m_fvg_list[];
   int               m_fvg_count;
   int               m_next_fvg_id;
   int               m_bars_total;
   string            m_symbol;
   ENUM_TIMEFRAMES   m_timeframe;
   ENUM_TIMEFRAMES   m_accuracy_tf;
   double            m_min_gap_size;
   bool              m_use_accuracy;
   bool              m_draw_objects;
   
public:
   CFVGLibrary(string symbol = "", ENUM_TIMEFRAMES timeframe = PERIOD_M5, 
               ENUM_TIMEFRAMES accuracy_tf = PERIOD_M1, double min_gap_size = 50.0,
               bool use_accuracy = false, bool draw_objects = true)
   {
      m_symbol = (symbol == "") ? _Symbol : symbol;
      m_timeframe = timeframe;
      m_accuracy_tf = accuracy_tf;
      m_min_gap_size = min_gap_size;
      m_use_accuracy = use_accuracy;
      m_draw_objects = draw_objects;
      m_fvg_count = 0;
      m_next_fvg_id = 1;
      m_bars_total = 0;
      ArrayResize(m_fvg_list, 100);
   }
   
   ~CFVGLibrary() { Deinit(); }
   
   bool Init()
   {
      Print("=== FVG Library Initialized ===");
      Print("Symbol: ", m_symbol);
      Print("Timeframe: ", EnumToString(m_timeframe));
      Print("Minimum Gap Size: ", m_min_gap_size, " points");
      
      if(m_draw_objects) CleanupObjects();
      
      m_fvg_count = 0;
      m_next_fvg_id = 1;
      m_bars_total = iBars(m_symbol, m_timeframe);
      
      return true;
   }
   
   void Deinit()
   {
      if(m_draw_objects) CleanupObjects();
      Print("=== FVG Library Deinitialized ===");
   }
   
   void OnTick()
   {
      int current_bars = iBars(m_symbol, m_timeframe);
      if(current_bars > m_bars_total)
      {
         m_bars_total = current_bars;
         CheckForNewFVG();
      }
      
      CheckForInverseFVG();
      
      if(m_use_accuracy)
         CheckFVGEncroachmentHighAccuracy();
      else
         CheckFVGEncroachment();
   }
   
   void CheckForNewFVG()
   {
      int bars_available = iBars(m_symbol, m_timeframe);
      if(bars_available < 4) return;
      
      double high1 = iHigh(m_symbol, m_timeframe, 3);
      double low1 = iLow(m_symbol, m_timeframe, 3);
      double high2 = iHigh(m_symbol, m_timeframe, 2);
      double low2 = iLow(m_symbol, m_timeframe, 2);
      double high3 = iHigh(m_symbol, m_timeframe, 1);
      double low3 = iLow(m_symbol, m_timeframe, 1);
      
      datetime middle_candle_time = iTime(m_symbol, m_timeframe, 2);
      double point = SymbolInfoDouble(m_symbol, SYMBOL_POINT);
      double min_gap_price = m_min_gap_size * point;
      
      if(high1 < low3)
      {
         double gap_size = low3 - high1;
         if(gap_size >= min_gap_price)
            CreateFVG(2, middle_candle_time, low3, high1, gap_size, true);
      }
      else if(low1 > high3)
      {
         double gap_size = low1 - high3;
         if(gap_size >= min_gap_price)
            CreateFVG(2, middle_candle_time, low1, high3, gap_size, false);
      }
   }
   
   bool CreateFVG(int middle_bar, datetime candle_time, double top, double bottom, double gap_size, bool is_bullish)
   {
      if(m_fvg_count >= ArraySize(m_fvg_list))
         ArrayResize(m_fvg_list, ArraySize(m_fvg_list) + 50);
      
      double enc_point = (top + bottom) / 2.0;
      double point = SymbolInfoDouble(m_symbol, SYMBOL_POINT);
      
      m_fvg_list[m_fvg_count].fvg_id = m_next_fvg_id;
      m_fvg_list[m_fvg_count].middle_candle_bar = middle_bar;
      m_fvg_list[m_fvg_count].candle_time = candle_time;
      m_fvg_list[m_fvg_count].top_price = top;
      m_fvg_list[m_fvg_count].bottom_price = bottom;
      m_fvg_list[m_fvg_count].enc_point = enc_point;
      m_fvg_list[m_fvg_count].gap_size = gap_size / point;
      m_fvg_list[m_fvg_count].is_bullish = is_bullish;
      m_fvg_list[m_fvg_count].is_active = true;
      m_fvg_list[m_fvg_count].enc_touched = false;
      m_fvg_list[m_fvg_count].directional_bias = "PENDING";
      m_fvg_list[m_fvg_count].trade_ticket = 0;
      m_fvg_list[m_fvg_count].trade_opened = false;
      m_fvg_list[m_fvg_count].enc_time = 0;
      m_fvg_list[m_fvg_count].is_inverse = false;
      m_fvg_list[m_fvg_count].is_reset = false;
      
      if(m_draw_objects)
         PlotFVGDot(middle_bar, candle_time, is_bullish, m_next_fvg_id);
      
      Print(">>> FVG ", m_next_fvg_id, " formed! Type: ", (is_bullish ? "BULLISH" : "BEARISH"));
      
      m_fvg_count++;
      m_next_fvg_id++;
      return true;
   }
   
   void CheckFVGEncroachment()
   {
      double current_high = iHigh(m_symbol, m_timeframe, 0);
      double current_low = iLow(m_symbol, m_timeframe, 0);
      double current_close = iClose(m_symbol, m_timeframe, 0);
      
      for(int i = 0; i < m_fvg_count; i++)
      {
         if(!m_fvg_list[i].is_active || m_fvg_list[i].enc_touched) continue;
         
         bool enc_touched_now = false;
         string bias = "PENDING";
         
         if(m_fvg_list[i].is_bullish)
         {
            if(current_low <= m_fvg_list[i].enc_point && current_high >= m_fvg_list[i].enc_point)
            {
               enc_touched_now = true;
               bias = (current_close > m_fvg_list[i].enc_point) ? "BULLISH" : "BEARISH";
            }
         }
         else
         {
            if(current_high >= m_fvg_list[i].enc_point && current_low <= m_fvg_list[i].enc_point)
            {
               enc_touched_now = true;
               bias = (current_close < m_fvg_list[i].enc_point) ? "BEARISH" : "BULLISH";
            }
         }
         
         if(enc_touched_now)
         {
            m_fvg_list[i].enc_touched = true;
            m_fvg_list[i].is_active = false;
            m_fvg_list[i].directional_bias = bias;
            m_fvg_list[i].enc_time = TimeCurrent();
            
            if(m_draw_objects)
               PlotEncroachmentDot(m_fvg_list[i].fvg_id, bias);
            
            Print("*** FVG ", m_fvg_list[i].fvg_id, " ENCROACHED! Bias: ", bias);
         }
      }
   }
   
   void CheckFVGEncroachmentHighAccuracy()
   {
      double current_high = iHigh(m_symbol, m_accuracy_tf, 0);
      double current_low = iLow(m_symbol, m_accuracy_tf, 0);
      double current_close = iClose(m_symbol, m_accuracy_tf, 0);
      
      for(int i = 0; i < m_fvg_count; i++)
      {
         if(!m_fvg_list[i].is_active || m_fvg_list[i].enc_touched) continue;
         
         bool enc_touched_now = false;
         string bias = "PENDING";
         
         if(m_fvg_list[i].is_bullish)
         {
            if(current_low <= m_fvg_list[i].enc_point && current_high >= m_fvg_list[i].enc_point)
            {
               enc_touched_now = true;
               bias = (current_close > m_fvg_list[i].enc_point) ? "BULLISH" : "BEARISH";
            }
         }
         else
         {
            if(current_high >= m_fvg_list[i].enc_point && current_low <= m_fvg_list[i].enc_point)
            {
               enc_touched_now = true;
               bias = (current_close < m_fvg_list[i].enc_point) ? "BEARISH" : "BULLISH";
            }
         }
         
         if(enc_touched_now)
         {
            m_fvg_list[i].enc_touched = true;
            m_fvg_list[i].is_active = false;
            m_fvg_list[i].directional_bias = bias;
            m_fvg_list[i].enc_time = TimeCurrent();
            
            if(m_draw_objects)
               PlotEncroachmentDot(m_fvg_list[i].fvg_id, bias);
            
            Print("*** FVG ", m_fvg_list[i].fvg_id, " ENCROACHED (HA)! Bias: ", bias);
         }
      }
   }
   
   void CheckForInverseFVG()
   {
      double current_open = iOpen(m_symbol, m_timeframe, 0);
      double current_close = iClose(m_symbol, m_timeframe, 0);
      
      for(int i = 0; i < m_fvg_count; i++)
      {
         if(!m_fvg_list[i].is_active || m_fvg_list[i].enc_touched || m_fvg_list[i].is_reset) continue;
         
         bool is_inverse = false;
         
         if(m_fvg_list[i].is_bullish)
         {
            if(current_open < m_fvg_list[i].bottom_price && current_close > m_fvg_list[i].top_price)
               is_inverse = true;
         }
         else
         {
            if(current_open > m_fvg_list[i].top_price && current_close < m_fvg_list[i].bottom_price)
               is_inverse = true;
         }
         
         if(is_inverse)
         {
            m_fvg_list[i].is_reset = true;
            m_fvg_list[i].is_active = false;
            m_fvg_list[i].directional_bias = "RESET";
            m_fvg_list[i].enc_time = TimeCurrent();
            m_fvg_list[i].is_inverse = true;
            
            Print("*** FVG ", m_fvg_list[i].fvg_id, " RESET by INVERSE CANDLE! ***");
         }
      }
   }
   
   void PlotFVGDot(int bar_index, datetime candle_time, bool is_bullish, int fvg_id)
   {
      if(!m_draw_objects) return;
      
      string obj_name = "FVG_DOT_" + IntegerToString(fvg_id);
      double candle_high = iHigh(m_symbol, m_timeframe, bar_index);
      double candle_low = iLow(m_symbol, m_timeframe, bar_index);
      double point = SymbolInfoDouble(m_symbol, SYMBOL_POINT);
      double dot_price = is_bullish ? candle_high + (10 * point) : candle_low - (10 * point);
      color dot_color = is_bullish ? clrLimeGreen : clrRed;
      
      ObjectCreate(0, obj_name, OBJ_ARROW, 0, candle_time, dot_price);
      ObjectSetInteger(0, obj_name, OBJPROP_ARROWCODE, 159);
      ObjectSetInteger(0, obj_name, OBJPROP_COLOR, dot_color);
      ObjectSetInteger(0, obj_name, OBJPROP_WIDTH, 3);
      ObjectSetInteger(0, obj_name, OBJPROP_BACK, false);
      ObjectSetInteger(0, obj_name, OBJPROP_SELECTABLE, false);
   }
   
   void PlotEncroachmentDot(int fvg_id, string bias)
   {
      if(!m_draw_objects) return;
      
      string obj_name = "ENC_DOT_" + IntegerToString(fvg_id);
      datetime current_time = iTime(m_symbol, m_timeframe, 0);
      double candle_high = iHigh(m_symbol, m_timeframe, 0);
      double point = SymbolInfoDouble(m_symbol, SYMBOL_POINT);
      double dot_price = candle_high + (15 * point);
      color dot_color = (bias == "BULLISH") ? clrOrange : clrBlue;
      
      ObjectCreate(0, obj_name, OBJ_ARROW, 0, current_time, dot_price);
      ObjectSetInteger(0, obj_name, OBJPROP_ARROWCODE, 159);
      ObjectSetInteger(0, obj_name, OBJPROP_COLOR, dot_color);
      ObjectSetInteger(0, obj_name, OBJPROP_WIDTH, 3);
      ObjectSetInteger(0, obj_name, OBJPROP_BACK, false);
   }
   
   void CleanupObjects()
   {
      ObjectsDeleteAll(0, "FVG_DOT_");
      ObjectsDeleteAll(0, "ENC_DOT_");
   }
};

//+------------------------------------------------------------------+
//| LIQUIDITY LIBRARY CLASS                                          |
//+------------------------------------------------------------------+
class CLiquidityLibrary
{
private:
   LiquidityInfo     m_liquidity_list[];
   SessionData       m_sessions[];
   DailyBiasInfo     m_daily_bias;
   int               m_liquidity_count;
   int               m_next_liq_id;
   string            m_symbol;
   ENUM_TIMEFRAMES   m_timeframe;
   int               m_gmt_offset;
   bool              m_draw_objects;
   double            m_equal_tolerance;
   int               m_swing_lookback;
   
public:
   CLiquidityLibrary(string symbol = "", ENUM_TIMEFRAMES timeframe = PERIOD_M5,
                    int gmt_offset = 0, bool draw_objects = true, double equal_tolerance = 5.0)
   {
      m_symbol = (symbol == "") ? _Symbol : symbol;
      m_timeframe = timeframe;
      m_gmt_offset = gmt_offset;
      m_draw_objects = draw_objects;
      m_equal_tolerance = equal_tolerance;
      m_liquidity_count = 0;
      m_next_liq_id = 1;
      m_swing_lookback = 3;
      ArrayResize(m_liquidity_list, 200);
      ArrayResize(m_sessions, 7);
   }
   
   ~CLiquidityLibrary() { Deinit(); }
   
   bool Init()
   {
      Print("=== Liquidity Library Initialized ===");
      Print("Symbol: ", m_symbol);
      Print("Timeframe: ", EnumToString(m_timeframe));
      
      if(m_draw_objects) CleanupObjects();
      UpdateSessions();
      UpdateDailyBias();
      return true;
   }
   
   void Deinit()
   {
      if(m_draw_objects) CleanupObjects();
      Print("=== Liquidity Library Deinitialized ===");
   }
   
   void OnTick()
   {
      UpdateSessions();
      CheckLiquiditySweeps();
      UpdateLiquidityStatus();
   }
   
   void OnNewBar()
   {
      DetectSwingPoints();
      DetectEqualLevels();
      UpdateDailyBias();
   }
   
   string GetDailyBias() const { return m_daily_bias.daily_bias; }
   string GetCurrentDOL() const { return m_daily_bias.current_dol; }
   
   void UpdateDailyBias()
   {
      CalculatePreviousDayHighLow();
      CalculatePreviousWeekHighLow();
      m_daily_bias.current_dol = DetermineDOL();
      
      if(StringFind(m_daily_bias.current_dol, "HIGH") != -1)
         m_daily_bias.daily_bias = "BULLISH";
      else if(StringFind(m_daily_bias.current_dol, "LOW") != -1)
         m_daily_bias.daily_bias = "BEARISH";
      else
         m_daily_bias.daily_bias = "NEUTRAL";
   }
   
   void CalculatePreviousDayHighLow()
   {
      m_daily_bias.pdh_level = iHigh(m_symbol, PERIOD_D1, 1);
      m_daily_bias.pdl_level = iLow(m_symbol, PERIOD_D1, 1);
      
      datetime yesterday = iTime(m_symbol, PERIOD_D1, 1);
      CreateLiquidity(LIQUIDITY_PDH, m_daily_bias.pdh_level, yesterday);
      CreateLiquidity(LIQUIDITY_PDL, m_daily_bias.pdl_level, yesterday);
   }
   
   void CalculatePreviousWeekHighLow()
   {
      m_daily_bias.pwh_level = iHigh(m_symbol, PERIOD_W1, 1);
      m_daily_bias.pwl_level = iLow(m_symbol, PERIOD_W1, 1);
      
      datetime last_week = iTime(m_symbol, PERIOD_W1, 1);
      CreateLiquidity(LIQUIDITY_PWH, m_daily_bias.pwh_level, last_week);
      CreateLiquidity(LIQUIDITY_PWL, m_daily_bias.pwl_level, last_week);
   }
   
   string DetermineDOL()
   {
      double current_price = iClose(m_symbol, m_timeframe, 0);
      double current_open = iOpen(m_symbol, PERIOD_D1, 0);
      
      double pdh_distance = MathAbs(current_open - m_daily_bias.pdh_level);
      double pdl_distance = MathAbs(current_open - m_daily_bias.pdl_level);
      
      if(pdh_distance < pdl_distance)
         return "PDH";
      else
         return "PDL";
   }
   
   bool CreateLiquidity(ENUM_LIQUIDITY_TYPE type, double price, datetime time, string session = "")
   {
      if(m_liquidity_count >= ArraySize(m_liquidity_list))
         ArrayResize(m_liquidity_list, ArraySize(m_liquidity_list) + 100);
      
      double point = SymbolInfoDouble(m_symbol, SYMBOL_POINT);
      double tolerance = m_equal_tolerance * point;
      
      for(int i = 0; i < m_liquidity_count; i++)
      {
         if(m_liquidity_list[i].type == type && 
            MathAbs(m_liquidity_list[i].price_level - price) < tolerance)
            return false;
      }
      
      m_liquidity_list[m_liquidity_count].liquidity_id = m_next_liq_id;
      m_liquidity_list[m_liquidity_count].type = type;
      m_liquidity_list[m_liquidity_count].time_created = time;
      m_liquidity_list[m_liquidity_count].price_level = price;
      m_liquidity_list[m_liquidity_count].is_swept = false;
      m_liquidity_list[m_liquidity_count].is_active = true;
      m_liquidity_list[m_liquidity_count].session_origin = session;
      m_liquidity_list[m_liquidity_count].equal_count = 1;
      m_liquidity_list[m_liquidity_count].tolerance = tolerance;
      
      if(m_draw_objects) PlotLiquidityLevel(m_liquidity_list[m_liquidity_count]);
      
      m_liquidity_count++;
      m_next_liq_id++;
      return true;
   }
   
   void DetectSwingPoints()
   {
      int bars_available = iBars(m_symbol, m_timeframe);
      if(bars_available < m_swing_lookback * 2 + 1) return;
      
      if(IsSwingHigh(1))
      {
         double high_price = iHigh(m_symbol, m_timeframe, 1);
         datetime high_time = iTime(m_symbol, m_timeframe, 1);
         CreateLiquidity(LIQUIDITY_SWING_HIGH, high_price, high_time);
      }
      
      if(IsSwingLow(1))
      {
         double low_price = iLow(m_symbol, m_timeframe, 1);
         datetime low_time = iTime(m_symbol, m_timeframe, 1);
         CreateLiquidity(LIQUIDITY_SWING_LOW, low_price, low_time);
      }
   }
   
   bool IsSwingHigh(int bar_index)
   {
      if(bar_index < m_swing_lookback || bar_index >= iBars(m_symbol, m_timeframe) - m_swing_lookback)
         return false;
      
      double center_high = iHigh(m_symbol, m_timeframe, bar_index);
      
      for(int i = 1; i <= m_swing_lookback; i++)
      {
         if(iHigh(m_symbol, m_timeframe, bar_index + i) >= center_high)
            return false;
      }
      
      for(int i = 1; i <= m_swing_lookback; i++)
      {
         if(iHigh(m_symbol, m_timeframe, bar_index - i) >= center_high)
            return false;
      }
      
      return true;
   }
   
   bool IsSwingLow(int bar_index)
   {
      if(bar_index < m_swing_lookback || bar_index >= iBars(m_symbol, m_timeframe) - m_swing_lookback)
         return false;
      
      double center_low = iLow(m_symbol, m_timeframe, bar_index);
      
      for(int i = 1; i <= m_swing_lookback; i++)
      {
         if(iLow(m_symbol, m_timeframe, bar_index + i) <= center_low)
            return false;
      }
      
      for(int i = 1; i <= m_swing_lookback; i++)
      {
         if(iLow(m_symbol, m_timeframe, bar_index - i) <= center_low)
            return false;
      }
      
      return true;
   }
   
   void CheckLiquiditySweeps()
   {
      double current_high = iHigh(m_symbol, m_timeframe, 0);
      double current_low = iLow(m_symbol, m_timeframe, 0);
      
      for(int i = 0; i < m_liquidity_count; i++)
      {
         if(!m_liquidity_list[i].is_active || m_liquidity_list[i].is_swept) continue;
         
         bool swept = false;
         
         if((m_liquidity_list[i].type == LIQUIDITY_SWING_HIGH || 
             m_liquidity_list[i].type == LIQUIDITY_EQUAL_HIGHS ||
             m_liquidity_list[i].type == LIQUIDITY_PDH ||
             m_liquidity_list[i].type == LIQUIDITY_PWH) && 
            current_high > m_liquidity_list[i].price_level)
         {
            swept = true;
         }
         else if((m_liquidity_list[i].type == LIQUIDITY_SWING_LOW ||
                  m_liquidity_list[i].type == LIQUIDITY_EQUAL_LOWS ||
                  m_liquidity_list[i].type == LIQUIDITY_PDL ||
                  m_liquidity_list[i].type == LIQUIDITY_PWL) && 
                 current_low < m_liquidity_list[i].price_level)
         {
            swept = true;
         }
         
         if(swept)
         {
            m_liquidity_list[i].is_swept = true;
            m_liquidity_list[i].sweep_time = TimeCurrent();
            if(m_draw_objects) PlotSweptLevel(m_liquidity_list[i]);
         }
      }
   }
   
   void DetectEqualLevels()
   {
      double point = SymbolInfoDouble(m_symbol, SYMBOL_POINT);
      double tolerance = m_equal_tolerance * point;
      
      for(int i = 0; i < m_liquidity_count - 1; i++)
      {
         if(!m_liquidity_list[i].is_active) continue;
         
         for(int j = i + 1; j < m_liquidity_count; j++)
         {
            if(!m_liquidity_list[j].is_active) continue;
            
            if(MathAbs(m_liquidity_list[i].price_level - m_liquidity_list[j].price_level) < tolerance)
            {
               bool same_category = false;
               
               if((m_liquidity_list[i].type == LIQUIDITY_SWING_HIGH || m_liquidity_list[i].type == LIQUIDITY_EQUAL_HIGHS) &&
                  (m_liquidity_list[j].type == LIQUIDITY_SWING_HIGH || m_liquidity_list[j].type == LIQUIDITY_EQUAL_HIGHS))
               {
                  same_category = true;
                  m_liquidity_list[i].type = LIQUIDITY_EQUAL_HIGHS;
                  m_liquidity_list[j].type = LIQUIDITY_EQUAL_HIGHS;
               }
               else if((m_liquidity_list[i].type == LIQUIDITY_SWING_LOW || m_liquidity_list[i].type == LIQUIDITY_EQUAL_LOWS) &&
                       (m_liquidity_list[j].type == LIQUIDITY_SWING_LOW || m_liquidity_list[j].type == LIQUIDITY_EQUAL_LOWS))
               {
                  same_category = true;
                  m_liquidity_list[i].type = LIQUIDITY_EQUAL_LOWS;
                  m_liquidity_list[j].type = LIQUIDITY_EQUAL_LOWS;
               }
               
               if(same_category)
               {
                  m_liquidity_list[i].is_equal_level = true;
                  m_liquidity_list[j].is_equal_level = true;
                  m_liquidity_list[i].equal_count++;
                  m_liquidity_list[j].equal_count = m_liquidity_list[i].equal_count;
               }
            }
         }
      }
   }
   
   void UpdateLiquidityStatus()
   {
      m_daily_bias.pdh_failure_to_displace = CheckFailureToDisplace(m_daily_bias.pdh_level, true);
      m_daily_bias.pdl_failure_to_displace = CheckFailureToDisplace(m_daily_bias.pdl_level, false);
      
      if(m_daily_bias.pdh_failure_to_displace)
         m_daily_bias.current_dol = "PDL";
      else if(m_daily_bias.pdl_failure_to_displace)
         m_daily_bias.current_dol = "PDH";
   }
   
   bool CheckFailureToDisplace(double level, bool is_high)
   {
      double current_high = iHigh(m_symbol, m_timeframe, 0);
      double current_low = iLow(m_symbol, m_timeframe, 0);
      double current_close = iClose(m_symbol, m_timeframe, 0);
      
      if(is_high)
      {
         if(current_high > level && current_close < level)
            return true;
      }
      else
      {
         if(current_low < level && current_close > level)
            return true;
      }
      
      return false;
   }
   
   void UpdateSessions()
   {
      datetime current_time = TimeCurrent() + (m_gmt_offset * 3600);
      MqlDateTime time_struct;
      TimeToStruct(current_time, time_struct);
      int current_hour = time_struct.hour;
      
      for(int i = 0; i < ArraySize(m_sessions); i++)
         m_sessions[i].is_active = false;
      
      if((current_hour >= 21) || (current_hour < 6))
      {
         m_sessions[SESSION_SYDNEY].is_active = true;
         m_sessions[SESSION_SYDNEY].session_name = "Sydney";
      }
      
      if(current_hour >= 0 && current_hour < 9)
      {
         m_sessions[SESSION_TOKYO].is_active = true;
         m_sessions[SESSION_TOKYO].session_name = "Tokyo";
      }
      
      if(current_hour >= 7 && current_hour < 16)
      {
         m_sessions[SESSION_LONDON].is_active = true;
         m_sessions[SESSION_LONDON].session_name = "London";
      }
      
      if(current_hour >= 12 && current_hour < 21)
      {
         m_sessions[SESSION_NEW_YORK].is_active = true;
         m_sessions[SESSION_NEW_YORK].session_name = "New York";
      }
   }
   
   void PlotLiquidityLevel(LiquidityInfo& liq)
   {
      if(!m_draw_objects) return;
      
      string obj_name = "LIQ_" + IntegerToString(liq.liquidity_id);
      color line_color = clrWhite;
      
      switch(liq.type)
      {
         case LIQUIDITY_SWING_HIGH:
         case LIQUIDITY_EQUAL_HIGHS:
            line_color = clrRed;
            break;
         case LIQUIDITY_SWING_LOW:
         case LIQUIDITY_EQUAL_LOWS:
            line_color = clrLimeGreen;
            break;
         case LIQUIDITY_PDH:
         case LIQUIDITY_PWH:
            line_color = clrOrange;
            break;
         case LIQUIDITY_PDL:
         case LIQUIDITY_PWL:
            line_color = clrAqua;
            break;
      }
      
      ObjectCreate(0, obj_name, OBJ_HLINE, 0, 0, liq.price_level);
      ObjectSetInteger(0, obj_name, OBJPROP_COLOR, line_color);
      ObjectSetInteger(0, obj_name, OBJPROP_STYLE, STYLE_DASH);
      ObjectSetInteger(0, obj_name, OBJPROP_WIDTH, 1);
      ObjectSetInteger(0, obj_name, OBJPROP_BACK, true);
   }
   
   void PlotSweptLevel(LiquidityInfo& liq)
   {
      if(!m_draw_objects) return;
      
      string obj_name = "LIQ_" + IntegerToString(liq.liquidity_id);
      ObjectSetInteger(0, obj_name, OBJPROP_COLOR, clrGray);
      ObjectSetInteger(0, obj_name, OBJPROP_STYLE, STYLE_DOT);
   }
   
   void CleanupObjects()
   {
      ObjectsDeleteAll(0, "LIQ_");
   }
};

//+------------------------------------------------------------------+
//| EA MAIN CODE                                                      |
//+------------------------------------------------------------------+

//--- Input parameters
input group "=== Trading Settings ==="
input double InpLotSize = 0.01;
input double InpProfitTarget = 2.0;
input bool   CheckMarginBeforeTrading = true;
input double MinimumFreeMarginPercent = 100.0;

input group "=== Library Settings ==="
input ENUM_TIMEFRAMES InpTimeframe = PERIOD_H1;
input int InpGMTOffset = 0;

//--- Global variables
CFVGLibrary* fvg_lib;
CLiquidityLibrary* liq_lib;
string daily_bias = "NEUTRAL";
bool trade_placed_today = false;
datetime current_day = 0;
datetime last_hourly_bar = 0;
ulong current_ticket = 0;
const int EA_MAGIC = 8888;

//+------------------------------------------------------------------+
//| Check if there's enough margin to open a trade                   |
//+------------------------------------------------------------------+
bool HasSufficientMargin(double lotSize)
{
   if(!CheckMarginBeforeTrading) return true;
   
   double normalizedLot = NormalizeLot(lotSize);
   if(normalizedLot <= 0) return false;
   
   double requiredMargin = 0;
   if(!OrderCalcMargin(ORDER_TYPE_BUY, _Symbol, normalizedLot, 
                       SymbolInfoDouble(_Symbol, SYMBOL_ASK), requiredMargin))
   {
      Print("Failed to calculate required margin. Error: ", GetLastError());
      return false;
   }
   
   double freeMargin = AccountInfoDouble(ACCOUNT_MARGIN_FREE);
   double equity = AccountInfoDouble(ACCOUNT_EQUITY);
   
   double marginAfterTrade = freeMargin - requiredMargin;
   double freeMarginPercent = (equity > 0) ? (marginAfterTrade / equity) * 100.0 : 0;
   
   if(freeMarginPercent < MinimumFreeMarginPercent)
   {
      Print("Insufficient margin to open trade. Required: ", requiredMargin, 
            " Free: ", freeMargin, " Free% after trade: ", freeMarginPercent,
            "% (minimum required: ", MinimumFreeMarginPercent, "%)");
      return false;
   }
   
   return true;
}

//+------------------------------------------------------------------+
//| Normalize lot size to broker requirements                        |
//+------------------------------------------------------------------+
double NormalizeLot(double lotSize)
{
   double minLot = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MIN);
   double maxLot = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MAX);
   double stepLot = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_STEP);
   
   if(lotSize < minLot) lotSize = minLot;
   if(lotSize > maxLot) lotSize = maxLot;
   
   lotSize = MathRound(lotSize / stepLot) * stepLot;
   int digits = (int)MathLog10(1.0 / stepLot);
   lotSize = NormalizeDouble(lotSize, digits);
   
   return lotSize;
}

//+------------------------------------------------------------------+
//| Expert initialization function                                    |
//+------------------------------------------------------------------+
int OnInit()
{
   Print("=== Hunter EA Initialized ===");

   fvg_lib = new CFVGLibrary(_Symbol, PERIOD_M5, PERIOD_M1, 50.0, false, false);
   if(!fvg_lib.Init())
   {
      Print("Failed to initialize FVG Library!");
      return(INIT_FAILED);
   }

   liq_lib = new CLiquidityLibrary(_Symbol, PERIOD_M5, InpGMTOffset, false);
   if(!liq_lib.Init())
   {
      Print("Failed to initialize Liquidity Library!");
      return(INIT_FAILED);
   }

   UpdateDailyBias();

   MqlDateTime time_struct;
   TimeToStruct(TimeCurrent(), time_struct);
   time_struct.hour = 0;
   time_struct.min = 0;
   time_struct.sec = 0;
   current_day = StructToTime(time_struct);

   last_hourly_bar = iTime(_Symbol, InpTimeframe, 0);

   Print("Initial Daily Bias: ", daily_bias);
   Print("Lot Size: ", InpLotSize);
   Print("===========================");

   return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
   if(fvg_lib != NULL)
   {
      fvg_lib.Deinit();
      delete fvg_lib;
      fvg_lib = NULL;
   }

   if(liq_lib != NULL)
   {
      liq_lib.Deinit();
      delete liq_lib;
      liq_lib = NULL;
   }

   Print("=== Hunter EA Deinitialized ===");
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
{
   if(fvg_lib != NULL) fvg_lib.OnTick();
   if(liq_lib != NULL) liq_lib.OnTick();

   CheckNewDay();
   CheckNewHourlyCandle();
   ManagePosition();
}

//+------------------------------------------------------------------+
//| Check for new day and reset                                      |
//+------------------------------------------------------------------+
void CheckNewDay()
{
   MqlDateTime time_struct;
   TimeToStruct(TimeCurrent(), time_struct);
   time_struct.hour = 0;
   time_struct.min = 0;
   time_struct.sec = 0;
   datetime today = StructToTime(time_struct);

   if(today > current_day)
   {
      current_day = today;
      trade_placed_today = false;
      UpdateDailyBias();

      Print("=== NEW DAY ===");
      Print("Date: ", TimeToString(current_day, TIME_DATE));
      Print("Daily Bias: ", daily_bias);
      Print("===============");
   }
}

//+------------------------------------------------------------------+
//| Update daily bias from Liquidity Library                        |
//+------------------------------------------------------------------+
void UpdateDailyBias()
{
   if(liq_lib == NULL) return;

   daily_bias = liq_lib.GetDailyBias();

   if(daily_bias == "NEUTRAL")
   {
      string dol = liq_lib.GetCurrentDOL();

      if(dol == "PDH" || dol == "PWH" || StringFind(dol, "HIGH") != -1)
         daily_bias = "BULLISH";
      else if(dol == "PDL" || dol == "PWL" || StringFind(dol, "LOW") != -1)
         daily_bias = "BEARISH";
   }
}

//+------------------------------------------------------------------+
//| Check for new hourly candle                                      |
//+------------------------------------------------------------------+
void CheckNewHourlyCandle()
{
   datetime current_bar = iTime(_Symbol, InpTimeframe, 0);

   if(current_bar > last_hourly_bar)
   {
      last_hourly_bar = current_bar;
      if(liq_lib != NULL) liq_lib.OnNewBar();
      TryPlaceTrade();
   }
}

//+------------------------------------------------------------------+
//| Try to place trade based on daily bias                          |
//+------------------------------------------------------------------+
void TryPlaceTrade()
{
   if(trade_placed_today)
   {
      Print("Trade already placed today - waiting for next day");
      return;
   }

   if(current_ticket != 0 && PositionSelectByTicket(current_ticket))
   {
      Print("Position still open - cannot place new trade");
      return;
   }

   if(daily_bias == "NEUTRAL")
   {
      Print("Daily bias is NEUTRAL - no trade");
      return;
   }

   ulong ticket = 0;

   if(daily_bias == "BULLISH")
   {
      ticket = OpenBuy(InpLotSize);
      if(ticket > 0)
      {
         current_ticket = ticket;
         trade_placed_today = true;
         Print("=== TRADE OPENED ===");
         Print("Direction: BUY");
         Print("Ticket: ", ticket);
         Print("====================");
      }
   }
   else if(daily_bias == "BEARISH")
   {
      ticket = OpenSell(InpLotSize);
      if(ticket > 0)
      {
         current_ticket = ticket;
         trade_placed_today = true;
         Print("=== TRADE OPENED ===");
         Print("Direction: SELL");
         Print("Ticket: ", ticket);
         Print("====================");
      }
   }
}

//+------------------------------------------------------------------+
//| Manage open position                                             |
//+------------------------------------------------------------------+
void ManagePosition()
{
   if(current_ticket == 0) return;

   if(!PositionSelectByTicket(current_ticket))
   {
      Print("Position ", current_ticket, " closed");
      current_ticket = 0;
      return;
   }

   double profit = PositionGetDouble(POSITION_PROFIT);

   if(profit >= InpProfitTarget)
   {
      Print("Profit target reached: $", DoubleToString(profit, 2));

      if(PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY)
         CloseBuy(current_ticket);
      else
         CloseSell(current_ticket);

      current_ticket = 0;
   }
}

//+------------------------------------------------------------------+
//| Open Buy position                                                 |
//+------------------------------------------------------------------+
ulong OpenBuy(double lotSize)
{
   if(!HasSufficientMargin(lotSize))
   {
      Print("OpenBuy skipped: Insufficient margin in account.");
      return 0;
   }

   double normalizedLot = NormalizeLot(lotSize);
   
   if(normalizedLot <= 0)
   {
      Print("OpenBuy failed: Invalid lot size");
      return 0;
   }

   MqlTradeRequest request;
   MqlTradeResult result;
   ZeroMemory(request);
   ZeroMemory(result);

   request.action = TRADE_ACTION_DEAL;
   request.symbol = _Symbol;
   request.volume = normalizedLot;
   request.type = ORDER_TYPE_BUY;
   request.price = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
   request.deviation = 20;
   request.type_filling = ORDER_FILLING_FOK;
   request.magic = EA_MAGIC;

   if(OrderSend(request, result))
   {
      if(result.deal > 0)
      {
         Print("Buy order opened successfully");
         return result.deal;
      }
   }
   else
   {
      Print("Failed to open buy order. Error: ", GetLastError());
   }

   return 0;
}

//+------------------------------------------------------------------+
//| Open Sell position                                                |
//+------------------------------------------------------------------+
ulong OpenSell(double lotSize)
{
   if(!HasSufficientMargin(lotSize))
   {
      Print("OpenSell skipped: Insufficient margin in account.");
      return 0;
   }

   double normalizedLot = NormalizeLot(lotSize);
   
   if(normalizedLot <= 0)
   {
      Print("OpenSell failed: Invalid lot size");
      return 0;
   }

   MqlTradeRequest request;
   MqlTradeResult result;
   ZeroMemory(request);
   ZeroMemory(result);

   request.action = TRADE_ACTION_DEAL;
   request.symbol = _Symbol;
   request.volume = normalizedLot;
   request.type = ORDER_TYPE_SELL;
   request.price = SymbolInfoDouble(_Symbol, SYMBOL_BID);
   request.deviation = 20;
   request.type_filling = ORDER_FILLING_FOK;
   request.magic = EA_MAGIC;

   if(OrderSend(request, result))
   {
      if(result.deal > 0)
      {
         Print("Sell order opened successfully");
         return result.deal;
      }
   }
   else
   {
      Print("Failed to open sell order. Error: ", GetLastError());
   }

   return 0;
}

//+------------------------------------------------------------------+
//| Close Buy position                                                |
//+------------------------------------------------------------------+
void CloseBuy(ulong ticket)
{
   if(!PositionSelectByTicket(ticket)) return;

   double volume = PositionGetDouble(POSITION_VOLUME);
   double price = SymbolInfoDouble(_Symbol, SYMBOL_BID);

   MqlTradeRequest request;
   MqlTradeResult result;
   ZeroMemory(request);
   ZeroMemory(result);

   request.action = TRADE_ACTION_DEAL;
   request.symbol = _Symbol;
   request.volume = volume;
   request.type = ORDER_TYPE_SELL;
   request.price = price;
   request.deviation = 20;
   request.type_filling = ORDER_FILLING_FOK;
   request.position = ticket;
   request.magic = EA_MAGIC;

   if(OrderSend(request, result))
   {
      Print("Buy position closed successfully");
   }
   else
   {
      Print("Failed to close buy position. Error: ", GetLastError());
   }
}

//+------------------------------------------------------------------+
//| Close Sell position                                               |
//+------------------------------------------------------------------+
void CloseSell(ulong ticket)
{
   if(!PositionSelectByTicket(ticket)) return;

   double volume = PositionGetDouble(POSITION_VOLUME);
   double price = SymbolInfoDouble(_Symbol, SYMBOL_ASK);

   MqlTradeRequest request;
   MqlTradeResult result;
   ZeroMemory(request);
   ZeroMemory(result);

   request.action = TRADE_ACTION_DEAL;
   request.symbol = _Symbol;
   request.volume = volume;
   request.type = ORDER_TYPE_BUY;
   request.price = price;
   request.deviation = 20;
   request.type_filling = ORDER_FILLING_FOK;
   request.position = ticket;
   request.magic = EA_MAGIC;

   if(OrderSend(request, result))
   {
      Print("Sell position closed successfully");
   }
   else
   {
      Print("Failed to close sell position. Error: ", GetLastError());
   }
}
//+------------------------------------------------------------------+